// Komunikacija po protokolu gRPC
// strežnik

package main

import (
	"api/grpc/protobufStorage"
	"api/storage"
	"context"
	"fmt"
	"net"
	"os"
	"google.golang.org/grpc"
	"google.golang.org/protobuf/types/known/emptypb"
	"sync"
	"log"
)

func Server(url string) {
	// pripravimo strežnik gRPC
	grpcServer := grpc.NewServer()

	// pripravimo strukturo za streženje metod CRUD na shrambi TodoStorage
	crudServer := NewServerCRUD()

	// streženje metod CRUD na shrambi TodoStorage povežemo s strežnikom gRPC
	protobufStorage.RegisterCRUDServer(grpcServer, crudServer)

	// izpišemo ime strežnika
	hostName, err := os.Hostname()
	if err != nil {
		panic(err)
	}
	// odpremo vtičnico
	listener, err := net.Listen("tcp", url)
	if err != nil {
		panic(err)
	}
	fmt.Printf("gRPC server listening at %v%v\n", hostName, url)
	// začnemo s streženjem
	if err := grpcServer.Serve(listener); err != nil {
		panic(err)
	}
}

// struktura za strežnik CRUD za shrambo TodoStorage
type serverCRUD struct {
	protobufStorage.UnimplementedCRUDServer
	todoStore *storage.TodoStorage
	naročniki map[chan protobufStorage.TodoEvent]struct{}
	naročniki_lock sync.RWMutex
}

// pripravimo nov strežnik CRUD za shrambo TodoStorage
func NewServerCRUD() *serverCRUD {
	todoStorePtr := storage.NewTodoStorage()
	return &serverCRUD{protobufStorage.UnimplementedCRUDServer{}, todoStorePtr, make(map[chan protobufStorage.TodoEvent]struct{}), sync.RWMutex{}}
}
func obvesti_naročnike (s *serverCRUD, dogodek protobufStorage.TodoEvent) {
	log.Println("obvesti_naročnike", dogodek)
	s.naročniki_lock.Lock()
	log.Println("obvesti_naročnike dobil ključavnico", dogodek)
	defer s.naročniki_lock.Unlock()
	naročniki_kopija := make(map[chan protobufStorage.TodoEvent]struct{})
	for k, v := range s.naročniki {
		naročniki_kopija[k] = v
	}
	for naročnik := range naročniki_kopija {
		select {
			case naročnik <- dogodek:
			default:
				log.Println("odklapljam naročnika, ker ima poln medpomnilnik")
				delete(s.naročniki, naročnik)
				close(naročnik)
		}
	}
}
// metode strežnika CRUD za shrambo TodoStorage
// odtis metode (argumenti, vrnjene vrednosti) so podane tako, kot zahtevajo paketi, ki jih je ustvaril prevajalnik protoc
// vse metode so samo ovojnice za klic metod v TodoStorage
// metode se izvajajo v okviru omejitev izvajalnega okolju ctx
func (s *serverCRUD) Create(ctx context.Context, in *protobufStorage.Todo) (*emptypb.Empty, error) {
	var ret struct{}
	log.Println("ustvarjam zapis", in)
	err := s.todoStore.Create(&storage.Todo{Task: in.Task, Completed: in.Completed}, &ret)
	log.Println("obveščam naročnike za ustvarjen zapis", in)
	if err == nil {
		obvesti_naročnike(s, protobufStorage.TodoEvent{T: in, Action: "create"})
	}
	log.Println("obvestil naročnike za ustvarjen zapis", in)
	return &emptypb.Empty{}, err
}

func (s *serverCRUD) Read(ctx context.Context, in *protobufStorage.Todo) (*protobufStorage.TodoStorage, error) {
	dict := make(map[string](storage.Todo))
	err := s.todoStore.Read(&storage.Todo{Task: in.Task, Completed: in.Completed}, &dict)
	pbDict := protobufStorage.TodoStorage{}
	for k, v := range dict {
		pbDict.Todos = append(pbDict.Todos, &protobufStorage.Todo{Task: k, Completed: v.Completed})
	}
	return &pbDict, err
}

func (s *serverCRUD) Update(ctx context.Context, in *protobufStorage.Todo) (*emptypb.Empty, error) {
	var ret struct{}
	err := s.todoStore.Update(&storage.Todo{Task: in.Task, Completed: in.Completed}, &ret)
	if err == nil {
		obvesti_naročnike(s, protobufStorage.TodoEvent{T: in, Action: "update"})
	}
	return &emptypb.Empty{}, err
}

func (s *serverCRUD) Delete(ctx context.Context, in *protobufStorage.Todo) (*emptypb.Empty, error) {
	var ret struct{}
	err := s.todoStore.Delete(&storage.Todo{Task: in.Task, Completed: in.Completed}, &ret)
	if err == nil {
		obvesti_naročnike(s, protobufStorage.TodoEvent{T: in, Action: "delete"})
	}
	return &emptypb.Empty{}, err
}
func (s *serverCRUD) ReadAll(e *emptypb.Empty, stream grpc.ServerStreamingServer[protobufStorage.Todo]) error {
	dict := make(map[string](storage.Todo))
	err := s.todoStore.Read(&storage.Todo{}, &dict)
	for _, todo := range dict {
		todo_temp := &protobufStorage.Todo{Task: todo.Task, Completed: todo.Completed}
		if err := stream.Send(todo_temp); err != nil {
			return err
		}
	}
	return err
}
func (s *serverCRUD) Subscribe (e *emptypb.Empty, stream grpc.ServerStreamingServer[protobufStorage.TodoEvent]) error {
	s.naročniki_lock.Lock()
	naročnina := make(chan protobufStorage.TodoEvent, 16)
	s.naročniki[naročnina] = struct{}{}
	s.naročniki_lock.Unlock()
	defer func () {
		s.naročniki_lock.Lock()
		delete(s.naročniki, naročnina)
		s.naročniki_lock.Unlock()
		close(naročnina)
	}()
	log.Println("začenjam range naročnina subscribe loop")
	for event := range naročnina {
		if err := stream.Send(&event); err != nil {
			return err
		}
	}
	log.Println("konec range naročnina subscribe loop")
	return nil
}
